// Store current CSP state
let currentCSP = null;
let isEnabled = false;

// Store CSP headers from pages we've seen
const pageCSPs = new Map();

// Listen for messages from popup and content scripts
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === 'updateCSP') {
        currentCSP = message.csp;
        isEnabled = message.enabled;
        
        // Update declarativeNetRequest rules
        updateRules();
        
        sendResponse({ success: true });
    } else if (message.action === 'getCurrentPageCSP') {
        // Get CSP for current page
        const tabId = message.tabId;
        const csp = pageCSPs.get(tabId) || null;
        sendResponse({ csp: csp });
    } else if (message.action === 'setPageCSP' && sender.tab) {
        // CSP from content script (meta tags)
        pageCSPs.set(sender.tab.id, message.csp);
        sendResponse({ success: true });
    }
    return true;
});

// Listen for web requests to capture CSP headers
chrome.webRequest.onHeadersReceived.addListener(
    (details) => {
        if (details.type === 'main_frame' || details.type === 'sub_frame') {
            // Look for CSP header
            const cspHeader = details.responseHeaders?.find(
                header => header.name.toLowerCase() === 'content-security-policy'
            );
            
            if (cspHeader) {
                // Store CSP for this tab
                pageCSPs.set(details.tabId, cspHeader.value);
            }
        }
    },
    {
        urls: ['<all_urls>'],
        types: ['main_frame', 'sub_frame']
    },
    ['responseHeaders', 'extraHeaders']
);


// Load saved state on startup
chrome.storage.sync.get(['cspEnabled', 'cspValue'], (result) => {
    if (result.cspEnabled && result.cspValue) {
        currentCSP = result.cspValue;
        isEnabled = true;
        updateRules();
    }
});

// Update declarativeNetRequest rules
async function updateRules() {
    // Remove existing rules
    const existingRules = await chrome.declarativeNetRequest.getDynamicRules();
    const ruleIds = existingRules.map(rule => rule.id);
    
    if (ruleIds.length > 0) {
        await chrome.declarativeNetRequest.updateDynamicRules({
            removeRuleIds: ruleIds
        });
    }
    
    // Add new rule if enabled
    if (isEnabled && currentCSP) {
        await chrome.declarativeNetRequest.updateDynamicRules({
            addRules: [{
                id: 1,
                priority: 1,
                action: {
                    type: 'modifyHeaders',
                    responseHeaders: [{
                        operation: 'set',
                        header: 'Content-Security-Policy',
                        value: currentCSP
                    }]
                },
                condition: {
                    resourceTypes: ['main_frame', 'sub_frame', 'stylesheet', 'script', 'image', 'font', 'object', 'xmlhttprequest', 'websocket', 'media', 'websocket', 'other']
                }
            }]
        });
    }
}

// Listen for storage changes (sync across devices)
chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'sync') {
        if (changes.cspEnabled || changes.cspValue) {
            chrome.storage.sync.get(['cspEnabled', 'cspValue'], (result) => {
                currentCSP = result.cspValue || null;
                isEnabled = result.cspEnabled || false;
                updateRules();
            });
        }
    }
});

