// CSP Directive explanations
const cspDirectiveDescriptions = {
    'default-src': {
        name: 'Default Source',
        description: 'The default rule for all content types. If you don\'t specify a rule for a specific content type, the browser will use this default rule.'
    },
    'script-src': {
        name: 'Script Source',
        description: 'Controls which JavaScript files can run on the website. This is one of the most important security settings for preventing XSS attacks.'
    },
    'style-src': {
        name: 'Style Source',
        description: 'Determines which stylesheets (CSS files) can be applied to the website. Prevents attackers from injecting malicious styles.'
    },
    'img-src': {
        name: 'Image Source',
        description: 'Specifies which sources can provide images. Prevents unauthorized images from being displayed, which could be used for tracking or phishing.'
    },
    'font-src': {
        name: 'Font Source',
        description: 'Controls where fonts can be loaded from. Ensures only trusted fonts are used, preventing potential security issues with font files.'
    },
    'connect-src': {
        name: 'Connect Source',
        description: 'Restricts which URLs the website can make network requests to (AJAX, fetch, WebSocket). Prevents sending sensitive data to malicious servers.'
    },
    'frame-src': {
        name: 'Frame Source',
        description: 'Specifies which websites can be embedded in frames (iframes). Prevents malicious websites from being embedded in pages.'
    },
    'object-src': {
        name: 'Object Source',
        description: 'Controls which sources can provide plugins like Flash or Java applets. Most modern websites should set this to \'none\'.'
    },
    'media-src': {
        name: 'Media Source',
        description: 'Determines which sources can provide audio and video files. Ensures only trusted media content is played.'
    },
    'frame-ancestors': {
        name: 'Frame Ancestors',
        description: 'Controls which websites can embed this page inside their frames. Prevents clickjacking attacks. Should typically be \'none\'.'
    },
    'base-uri': {
        name: 'Base URI',
        description: 'Restricts which URLs can be used as the base URL for relative links. Prevents attackers from changing where links point.'
    },
    'form-action': {
        name: 'Form Action',
        description: 'Controls which URLs can receive form submissions. Prevents forms from sending data to malicious servers.'
    },
    'upgrade-insecure-requests': {
        name: 'Upgrade Insecure Requests',
        description: 'Automatically upgrades all HTTP requests to HTTPS. Ensures all connections are encrypted.'
    },
    'block-all-mixed-content': {
        name: 'Block All Mixed Content',
        description: 'Blocks all mixed content (HTTP resources on HTTPS pages). Prevents insecure content from loading on secure pages.'
    }
};

// Parse CSP string into directives
function parseCSP(cspString) {
    if (!cspString || !cspString.trim()) {
        return [];
    }
    
    const directives = [];
    const parts = cspString.split(';').map(p => p.trim()).filter(p => p.length > 0);
    
    parts.forEach(part => {
        const spaceIndex = part.indexOf(' ');
        if (spaceIndex === -1) {
            // Directive without value (like upgrade-insecure-requests)
            directives.push({
                name: part,
                value: '',
                full: part
            });
        } else {
            const name = part.substring(0, spaceIndex);
            const value = part.substring(spaceIndex + 1).trim();
            directives.push({
                name: name,
                value: value,
                full: part
            });
        }
    });
    
    return directives;
}

// Generate explanation HTML for CSP
function explainCSP(cspString) {
    const directives = parseCSP(cspString);
    
    if (directives.length === 0) {
        return '<p style="color: var(--text-gray); font-size: 0.85rem;">No CSP directives found.</p>';
    }
    
    let html = '<div class="csp-explanation">';
    
    directives.forEach(dir => {
        const info = cspDirectiveDescriptions[dir.name] || {
            name: dir.name,
            description: 'Custom or unknown directive.'
        };
        
        html += `
            <div class="directive-item">
                <div class="directive-name">${info.name} (<code>${dir.name}</code>)</div>
                ${dir.value ? `<div class="directive-value">${escapeHtml(dir.value)}</div>` : ''}
                <div class="directive-description">${info.description}</div>
            </div>
        `;
    });
    
    html += '</div>';
    return html;
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Rate the security of a directive (0-5 scale)
function rateDirectiveSecurity(directive, value) {
    if (!value && !['upgrade-insecure-requests', 'block-all-mixed-content'].includes(directive)) return 0;
    
    // Special directives that don't need values
    if (directive === 'upgrade-insecure-requests' || directive === 'block-all-mixed-content') {
        return 5; // Best practice
    }
    
    const valueLower = value.toLowerCase();
    const values = valueLower.split(/\s+/).filter(v => v.length > 0);
    
    // Base rating for having the directive enabled
    let rating = 2;
    
    switch(directive) {
        case 'object-src':
            if (values.includes("'none'")) return 5; // Best practice
            if (values.includes("'self'")) return 3;
            return 1;
            
        case 'frame-ancestors':
            if (values.includes("'none'")) return 5; // Best practice
            if (values.includes("'self'")) return 4;
            return 2;
            
        case 'script-src':
            if (values.includes("'unsafe-inline'") || values.includes("'unsafe-eval'")) {
                rating = 1; // Very insecure
            } else if (values.includes("'none'")) {
                rating = 5; // Very secure but may break functionality
            } else if (values.includes("'self'") && !values.some(v => v.startsWith('https:') || v.startsWith('http:'))) {
                rating = 4; // Good - only self
            } else if (values.includes("'self'")) {
                rating = 3; // Moderate - self + external
            } else if (values.some(v => v.startsWith('https:'))) {
                rating = 2; // Permissive
            }
            return rating;
            
        case 'style-src':
            if (values.includes("'unsafe-inline'")) {
                rating = 2; // Less secure but often necessary
            } else if (values.includes("'none'")) {
                rating = 5;
            } else if (values.includes("'self'") && !values.some(v => v.startsWith('https:') || v.startsWith('http:'))) {
                rating = 4;
            } else if (values.includes("'self'")) {
                rating = 3;
            }
            return rating;
            
        case 'default-src':
            if (values.includes("'unsafe-inline'") || values.includes("'unsafe-eval'")) {
                rating = 1;
            } else if (values.includes("'none'")) {
                rating = 5;
            } else if (values.includes("'self'") && !values.some(v => v.startsWith('https:') || v.startsWith('http:'))) {
                rating = 4;
            } else if (values.includes("'self'")) {
                rating = 3;
            } else if (values.some(v => v.startsWith('https:'))) {
                rating = 2;
            }
            return rating;
            
        case 'img-src':
        case 'font-src':
        case 'media-src':
            if (values.includes("'none'")) return 5;
            if (values.includes("'self'") && !values.some(v => v.startsWith('https:') || v.startsWith('http:'))) return 4;
            if (values.includes("'self'")) return 3;
            if (values.some(v => v.startsWith('https:'))) return 2;
            return 2;
            
        case 'connect-src':
        case 'frame-src':
            if (values.includes("'none'")) return 5;
            if (values.includes("'self'") && !values.some(v => v.startsWith('https:') || v.startsWith('http:'))) return 4;
            if (values.includes("'self'")) return 3;
            if (values.some(v => v.startsWith('https:'))) return 2;
            return 2;
            
        case 'base-uri':
        case 'form-action':
            if (values.includes("'none'")) return 5;
            if (values.includes("'self'")) return 4;
            return 2;
            
        default:
            // Unknown directive - give base rating
            if (values.includes("'none'")) return 5;
            if (values.includes("'self'")) return 3;
            return 2;
    }
}

// Calculate overall security score from CSP
function calculateSecurityScore(cspString) {
    const directives = parseCSP(cspString);
    
    if (directives.length === 0) return null;
    
    // Critical directives that should be present for a secure CSP
    const criticalDirectives = ['default-src', 'script-src', 'object-src', 'frame-ancestors', 'base-uri', 'form-action'];
    const bestPracticeDirectives = ['upgrade-insecure-requests', 'block-all-mixed-content'];
    
    const directiveNames = directives.map(d => d.name);
    const directiveRatings = [];
    
    // Rate each configured directive
    directives.forEach(dir => {
        const rating = rateDirectiveSecurity(dir.name, dir.value);
        if (rating > 0) {
            directiveRatings.push({
                directive: dir.name,
                rating: rating
            });
        }
    });
    
    if (directiveRatings.length === 0) return null;
    
    const ratings = directiveRatings.map(dr => dr.rating);
    
    // Calculate average of configured directives
    const configuredAverage = ratings.reduce((sum, rating) => sum + rating, 0) / ratings.length;
    
    // Count missing critical directives
    const missingCritical = criticalDirectives.filter(d => !directiveNames.includes(d)).length;
    const missingBestPractice = bestPracticeDirectives.filter(d => !directiveNames.includes(d)).length;
    
    // Penalty for missing directives (more missing = lower score)
    // Each missing critical directive reduces score by 0.5
    // Each missing best practice reduces score by 0.2
    const penalty = (missingCritical * 0.5) + (missingBestPractice * 0.2);
    
    // Calculate final score (can't go below 0)
    const finalScore = Math.max(0, configuredAverage - penalty);
    
    // Additional penalty if very few directives configured
    // If only 1-2 directives, apply additional penalty
    if (directives.length <= 2) {
        const coveragePenalty = (3 - directives.length) * 0.8;
        return {
            average: Math.max(0, Math.round((finalScore - coveragePenalty) * 10) / 10),
            ratings: ratings,
            directiveRatings: directiveRatings,
            count: ratings.length,
            missingCritical: missingCritical,
            missingBestPractice: missingBestPractice
        };
    }
    
    return {
        average: Math.round(finalScore * 10) / 10, // Round to 1 decimal
        ratings: ratings,
        directiveRatings: directiveRatings,
        count: ratings.length,
        missingCritical: missingCritical,
        missingBestPractice: missingBestPractice
    };
}

// Get criticality rating based on score
function getCriticalityRating(score) {
    if (score >= 4.5) return { level: 'Excellent', color: '#4CAF50', description: 'Highly secure configuration' };
    if (score >= 3.5) return { level: 'Good', color: '#8BC34A', description: 'Secure configuration with good practices' };
    if (score >= 2.5) return { level: 'Moderate', color: '#FFC107', description: 'Moderate security - consider improvements' };
    if (score >= 1.5) return { level: 'Weak', color: '#FF9800', description: 'Weak security - significant improvements needed' };
    return { level: 'Critical', color: '#F44336', description: 'Critical security issues - immediate action required' };
}

