// Get DOM elements
const viewModeBtn = document.getElementById('view-mode-btn');
const testModeBtn = document.getElementById('test-mode-btn');
const viewModeContent = document.getElementById('view-mode-content');
const testModeContent = document.getElementById('test-mode-content');
const enableToggle = document.getElementById('enable-toggle');
const cspInput = document.getElementById('csp-input');
const pasteBtn = document.getElementById('paste-btn');
const clearBtn = document.getElementById('clear-btn');
const cspLoading = document.getElementById('csp-loading');
const cspNotFound = document.getElementById('csp-not-found');
const cspDisplay = document.getElementById('csp-display');
const cspRawText = document.getElementById('csp-raw-text');
const cspExplanation = document.getElementById('csp-explanation');
const securityScoreSection = document.getElementById('security-score-section');
const securityScoreValue = document.getElementById('security-score-value');
const securityScoreRating = document.getElementById('security-score-rating');
const securityScoreBreakdown = document.getElementById('security-score-breakdown');

// Load saved state
async function loadState() {
    const result = await chrome.storage.sync.get(['cspEnabled', 'cspValue']);
    
    if (result.cspEnabled) {
        enableToggle.checked = true;
    }
    
    if (result.cspValue) {
        cspInput.value = result.cspValue;
    }
    
    // Update background script
    updateCSPInjection();
}

// Save state
async function saveState() {
    await chrome.storage.sync.set({
        cspEnabled: enableToggle.checked,
        cspValue: cspInput.value.trim()
    });
    
    // Update background script
    updateCSPInjection();
}

// Update CSP injection in background script
async function updateCSPInjection() {
    const cspValue = cspInput.value.trim();
    const isEnabled = enableToggle.checked && cspValue.length > 0;
    
    // Send message to background script
    chrome.runtime.sendMessage({
        action: 'updateCSP',
        enabled: isEnabled,
        csp: cspValue
    });
}

// Event listeners
enableToggle.addEventListener('change', saveState);

cspInput.addEventListener('input', () => {
    saveState();
});

cspInput.addEventListener('blur', () => {
    saveState();
});

pasteBtn.addEventListener('click', async () => {
    try {
        const text = await navigator.clipboard.readText();
        cspInput.value = text;
        saveState();
        
        // Show feedback
        pasteBtn.textContent = '✓ Pasted!';
        setTimeout(() => {
            pasteBtn.textContent = 'Paste from Clipboard';
        }, 2000);
    } catch (err) {
        console.error('Failed to paste:', err);
        pasteBtn.textContent = '✗ Failed';
        setTimeout(() => {
            pasteBtn.textContent = 'Paste from Clipboard';
        }, 2000);
    }
});

clearBtn.addEventListener('click', () => {
    cspInput.value = '';
    saveState();
});

// Mode switching
viewModeBtn.addEventListener('click', () => {
    viewModeBtn.classList.add('active');
    testModeBtn.classList.remove('active');
    viewModeContent.classList.add('active');
    testModeContent.classList.remove('active');
    loadCurrentPageCSP();
});

testModeBtn.addEventListener('click', () => {
    testModeBtn.classList.add('active');
    viewModeBtn.classList.remove('active');
    viewModeContent.classList.remove('active');
    testModeContent.classList.add('active');
});

// Load current page CSP in view mode
async function loadCurrentPageCSP() {
    cspLoading.classList.remove('hidden');
    cspNotFound.classList.add('hidden');
    cspDisplay.classList.add('hidden');
    
    try {
        // Get current active tab
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        
        if (!tab || !tab.url || tab.url.startsWith('chrome://') || tab.url.startsWith('edge://') || tab.url.startsWith('about:')) {
            cspLoading.classList.add('hidden');
            cspNotFound.classList.remove('hidden');
            cspNotFound.querySelector('.info-text').innerHTML = '<strong>Cannot read CSP:</strong> This page is a browser internal page. Navigate to a regular website to view its CSP.';
            return;
        }
        
        // Try to get CSP from response headers via background script
        chrome.runtime.sendMessage({
            action: 'getCurrentPageCSP',
            tabId: tab.id
        }, (response) => {
            // If not found in headers, try to inject content script to check meta tags
            if (!response || !response.csp) {
                // Inject content script to check for meta tag CSP
                chrome.scripting.executeScript({
                    target: { tabId: tab.id },
                    function: getCSPFromMeta
                }, (results) => {
                    cspLoading.classList.add('hidden');
                    
                    if (results && results[0] && results[0].result) {
                        const csp = results[0].result;
                        displayCSP(csp);
                    } else {
                        // No CSP found
                        cspNotFound.classList.remove('hidden');
                        cspDisplay.classList.add('hidden');
                        securityScoreSection.classList.add('hidden');
                    }
                });
            } else {
                // CSP found in headers
                cspLoading.classList.add('hidden');
                displayCSP(response.csp);
            }
        });
    } catch (error) {
        console.error('Error loading CSP:', error);
        cspLoading.classList.add('hidden');
        cspNotFound.classList.remove('hidden');
        cspNotFound.querySelector('.info-text').innerHTML = '<strong>Error:</strong> Could not read CSP from this page.';
    }
}

// Function to get CSP from meta tag (injected into page)
function getCSPFromMeta() {
    const cspMeta = document.querySelector('meta[http-equiv="Content-Security-Policy"]');
    return cspMeta ? cspMeta.content : null;
}

// Display CSP with scoring
function displayCSP(csp) {
    cspLoading.classList.add('hidden');
    cspRawText.textContent = csp;
    cspExplanation.innerHTML = explainCSP(csp);
    cspDisplay.classList.remove('hidden');
    cspNotFound.classList.add('hidden');
    
    // Calculate and display security score
    const securityScore = calculateSecurityScore(csp);
    if (securityScore) {
        const criticality = getCriticalityRating(securityScore.average);
        securityScoreValue.textContent = `${securityScore.average}/5.0`;
        securityScoreValue.style.color = criticality.color;
        securityScoreRating.innerHTML = `<strong style="color: ${criticality.color}">${criticality.level}</strong> - ${criticality.description}`;
        
        // Display directive breakdown
        if (securityScore.directiveRatings && securityScore.directiveRatings.length > 0) {
            let breakdownHTML = '<div class="security-score-breakdown"><strong>Directive Scores:</strong><ul>';
            securityScore.directiveRatings.forEach(dr => {
                const dirCriticality = getCriticalityRating(dr.rating);
                breakdownHTML += `<li><span class="directive-name">${dr.directive}:</span> <span class="directive-score" style="color: ${dirCriticality.color}">${dr.rating}/5</span></li>`;
            });
            breakdownHTML += '</ul></div>';
            securityScoreBreakdown.innerHTML = breakdownHTML;
        } else {
            securityScoreBreakdown.innerHTML = '';
        }
        
        securityScoreSection.classList.remove('hidden');
    } else {
        securityScoreSection.classList.add('hidden');
    }
}

// Load state on popup open
loadState();

// Load CSP if in view mode
if (viewModeContent.classList.contains('active')) {
    loadCurrentPageCSP();
}

